﻿using System;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Threading.Tasks;
using System.Windows;
using Tessa.Extensions.WorkflowExamples.Shared.Workflow;
using Tessa.Platform.Collections;
using Tessa.Platform.Storage;
using Tessa.UI.Cards;
using Tessa.UI.Cards.Forms;
using Tessa.UI.Cards.Tasks;

namespace Tessa.Extensions.WorkflowExamples.Client.Workflow
{
    [SuppressMessage("ReSharper", "InconsistentNaming")]
    public class WfeCardUIExtension : CardUIExtension
    {
        public override async Task Initialized(ICardUIExtensionContext context)
        {
            if (context.Model.MainForm is DefaultFormTabWithTasksViewModel formTask)
            {
                foreach (TaskViewModel taskViewModel in formTask.Tasks.OfType<TaskViewModel>())
                {
                    await taskViewModel.ModifyWorkspaceAsync(ModifyTaskAndAttachHandlersAsync);
                }

                formTask.HiddenTaskCreated += async (s, e) =>
                {
                    var deferral = e.Defer();
                    try
                    {
                        await e.Task.ModifyWorkspaceAsync(ModifyTaskAndAttachHandlersAsync);
                    }
                    catch (Exception ex)
                    {
                        deferral.SetException(ex);
                    }
                    finally
                    {
                        deferral.Dispose();
                    }
                };
            }
        }

        private static ValueTask ModifyTaskAndAttachHandlersAsync(TaskViewModel taskViewModel, bool isFirst)
        {
            Guid taskTypeID = taskViewModel.TaskModel.CardType.ID;

            if (taskTypeID == WfeHelper.FillingApplicationTypeID
                || taskTypeID == WfeHelper.RequestConsultationTypeID)
            {
                RemoveTaskActionFillingApplicationTypeID(taskViewModel);
            }

            if (taskTypeID == WfeHelper.RequestExpertiseTypeID
                || taskTypeID == WfeHelper.RequestConsultationTypeID
                || taskTypeID == WfeHelper.ProjectApprovalTypeID
                || taskTypeID == WfeHelper.FillingApplicationTypeID
                || taskTypeID == WfeHelper.SelectSalesSupervisorTypeID)
            {
                ModifyResolutionTask(taskViewModel);
            }
            else if(taskTypeID == WfeHelper.PositiveDecisionTypeID)
            {
                ModifyResolutionTaskPe(taskViewModel);
            }
            else if(taskTypeID == WfeHelper.ExpertiseInDepartmentAppointExecutiveTypeID)
            {
                ModifyResolutionTaskTe(taskViewModel);
            }
            else if (taskTypeID == WfeHelper.ExpertiseInDepartmentTypeID)
            {
                RemoveTaskAction(taskViewModel);
            }

            return new ValueTask();
        }

        private static void ModifyResolutionTaskTe(TaskViewModel taskViewModel)
        {
            RemoveTaskAction(taskViewModel);

            if (WfeUIHelper.TasksEmployeeIsNullOrEmpty(taskViewModel))
            {
                HideTableTE(taskViewModel, WfeUIHelper.TasksEmployeeControlName);
            }
        }

        private static void RemoveTaskActionFillingApplicationTypeID(TaskViewModel taskViewModel)
        {
            var segment = taskViewModel.TaskModel.ParentModel.Card.Sections["WfeLACommonInfo"].Fields.TryGet<int>("SegmentID");

            if (segment == 2)
            {
                var comOption = WfeCompletionOptions.OnRequestExpertise;
                if (taskViewModel.Workspace.Actions.Any(x => x.CompletionOption != null && x.CompletionOption.ID == comOption))
                {
                    taskViewModel.Workspace.Actions.RemoveAll(x => x.CompletionOption != null && x.CompletionOption.ID == comOption);
                }
            }
        }

        private static void RemoveTaskAction(TaskViewModel taskViewModel)
        {
            var step = taskViewModel.TaskModel.ParentModel.Card.Sections["WfeLACommonInfo"].Fields.TryGet<int?>("ProcessStepID");

            // удаление вариантов завершения
            if (WfeUIHelper.HidenTaskInfo.TryGetValue(step ?? 0, out var taskInfo))
            {
                foreach (var comOption in taskInfo)
                {
                    if (taskViewModel.Workspace.Actions.Any(x => x.CompletionOption != null && x.CompletionOption.ID == comOption))
                    {
                        taskViewModel.Workspace.Actions.RemoveAll(x => x.CompletionOption != null && x.CompletionOption.ID == comOption);
                    }
                }
            }
        }

        private static void HideTable(TaskItemViewModel taskViewModel, string controlName)
        {
            IFormWithBlocksViewModel form = taskViewModel.Workspace.Form;

            IBlockViewModel requestExpertiseBlock;
            if (form != null && (requestExpertiseBlock = form.Blocks.FirstOrDefault(x => WfeUIHelper.IsHideBlock(x.Name))) != null)
            {
                IControlViewModel requestedExpertiseControl;
                if ((requestedExpertiseControl = requestExpertiseBlock.Form.Blocks[0].Controls.FirstOrDefault(x => x.Name == controlName)) != null)
                {
                    requestedExpertiseControl.ControlVisibility = Visibility.Collapsed;
                    form.RearrangeSelf();
                }
            }
        }

        private static void HideTableTE(TaskItemViewModel taskViewModel, string controlName)
        {
            IFormWithBlocksViewModel form = taskViewModel.Workspace.Form;

            IBlockViewModel requestExpertiseBlock;
            if (form != null && (requestExpertiseBlock = form.Blocks.FirstOrDefault(x => WfeUIHelper.IsHideBlock(x.Name))) != null)
            {
                IControlViewModel requestedExpertiseControl;
                if ((requestedExpertiseControl = requestExpertiseBlock.Form.Blocks[1].Controls.FirstOrDefault(x => x.Name == controlName)) != null)
                {
                    requestedExpertiseControl.ControlVisibility = Visibility.Collapsed;
                    form.RearrangeSelf();
                }
            }
        }

        private static void ModifyResolutionTask(TaskViewModel taskViewModel)
        {
            // в зависимости от типа задания, убираем не нужные варианты завершения.
            RemoveTaskAction(taskViewModel);

            // серываем таблици в экспертизами, если эти таблицы пусты.
            if (WfeUIHelper.ExpertiseRowsIsNullOrEmpty(taskViewModel))
            {
                HideTable(taskViewModel, WfeUIHelper.RequestedExpertiseControlName);
            }
        }

        private static void ModifyResolutionTaskPe(TaskViewModel taskViewModel)
        {
            // в зависимости от типа задания, убираем не нужные варианты завершения.
            RemoveTaskAction(taskViewModel);

            // серываем таблици в экспертизами, если эти таблицы пусты.
            if (WfeUIHelper.ExpertiseRowsIsNullOrEmpty(taskViewModel))
            {
                HideTable(taskViewModel, WfeUIHelper.RequestedExpertiseControlName);
            }
        }
    }
}
