﻿using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Tessa.Cards;
using Tessa.Cards.Extensions;
using Tessa.Platform.Data;
using Tessa.Platform.Validation;

namespace Tessa.Extensions.AclExamples.Server.Cards
{
    public class DepartmentRoleStoreExtension : CardStoreExtension
    {
        #region Fields

        private readonly ICardRepository cardRepository;

        #endregion

        #region Constructors

        public DepartmentRoleStoreExtension(ICardRepository cardRepository)
        {
            this.cardRepository = cardRepository;
        }

        #endregion

        #region Private Methods

        private void BuildParentDepartmentsQuery(IQueryBuilder builder, int? parentDepartmentType)
        {
            builder
                .With("DepartmentsCTE", dc =>
                        dc.Select()
                                .C("rc", "ID", "ParentID").V(0).As("HierarchyLevel")
                            .From("Roles", "rc").NoLock()
                            .Where()
                                .C("rc", "TypeID").Equals().V(2)
                                .And()
                                .C("rc", "ID").Equals().P("DepartmentdID")
                            .UnionAll()
                            .Select()
                                .C("rp", "ID", "ParentID").C("dc", "HierarchyLevel").Add().V(1).As("HierarchyLevel")
                            .From("Roles", "rp").NoLock()
                            .InnerJoin("DepartmentsCTE", "dc")
                                .On().C("dc", "ParentID").Equals().C("rp", "ID")
                            .Where()
                                .C("rp", "TypeID").Equals().V(2),
                    recursive: true)
                .Update("AclDepartmentRoles")
                    .C("MainDepartmentID").Assign().C("rs", "ID")
                    .C("MainDepartmentName").Assign().C("rs", "Name")
                .From(rs => rs
                    .Select().Top(1)
                        .C("r", "ID", "Name")
                    .From("DepartmentsCTE", "dc")
                    .InnerJoin("Roles", "r").NoLock()
                        .On().C("r", "ID").Equals().C("dc", "ID")
                    .InnerJoin("AclDepartmentRoles", "acd").NoLock()
                        .On().C("acd", "ID").Equals().C("r", "ID")
                    .Where()
                        .If(parentDepartmentType.HasValue, q => q.C("acd", "DepartmentTypeID").Equals().P("DepartmentTypeID"))
                        .If(!parentDepartmentType.HasValue, q => q.C("acd", "DepartmentTypeID").IsNotNull())
                    .OrderBy("dc", "HierarchyLevel", SortOrder.Ascending)
                    .Limit(1), "rs")
                .Where()
                    .C("AclDepartmentRoles", "ID").Equals().P("DepartmentdID");
        }
        private async Task<bool> UpdateCurrentDepartmentAsync(
            ICardStoreExtensionContext context,
            DbManager db, 
            Card card, 
            int? currentDepartmentTypeID)
        {
            var builder = context.DbScope.BuilderFactory.Create();
            int? mainParentDepartmentTypeID = null;

            switch (currentDepartmentTypeID)
            {
                case null:
                    // первый филиал, представительство или организация выше по дереву
                    this.BuildParentDepartmentsQuery(builder, null);
                    break;
                case (int)DepartmentType.Organization:
                    // Сам себе главный
                    builder
                        .Update("AclDepartmentRoles")
                            .C("MainDepartmentID").Assign().C("t", "ID")
                            .C("MainDepartmentName").Assign().C("t", "Name")
                        .From(q=>q
                            .Select()
                                .C("r", "ID", "Name")
                            .From("Roles", "r").NoLock()
                            .Where()
                                .C("r", "ID").Equals().P("DepartmentdID"), "t")
                        .Where()
                            .C("AclDepartmentRoles", "ID").Equals().P("DepartmentdID");
                    break;
                case (int)DepartmentType.Branch: // филиал
                    // первое представительство выше по дереву
                    this.BuildParentDepartmentsQuery(builder, 2);
                    mainParentDepartmentTypeID = 2;
                    break;
                case (int)DepartmentType.Agency: // Представительство
                    // первая организация выше по дереву
                    this.BuildParentDepartmentsQuery(builder, 0);
                    mainParentDepartmentTypeID = 0;
                    break;
                default:
                    context.ValidationResult.AddError(this, "$Messages_WrongDepartmentTypeID");
                    return false;
            }

            db
                .SetCommand(
                    builder.Build(),
                    db.Parameter("DepartmentdID", card.ID),
                    db.Parameter("DepartmentTypeID", mainParentDepartmentTypeID))
                .LogCommand();

            if (await db.ExecuteNonQueryAsync(context.CancellationToken) == 0)
            {
                context.ValidationResult.AddError(this, "$Messages_CantUpdateMainDepartmentInfo");
                return false;
            }

            return true;
        }

        private async Task<(Guid? mainDepartmentID, string mainDepartmentName)> GetMainDepartmentInfoAsync(
            ICardStoreExtensionContext context, 
            DbManager db, 
            Guid departmentID, 
            int? currentDepartmentTypeID)
        {
            var builder = context.DbScope.BuilderFactory.Create();
            int? mainParentDepartmentTypeID = null;

            switch (currentDepartmentTypeID)
            {
                case null:
                    // первый филиал, представительство или организация выше по дереву
                    break;
                case (int)DepartmentType.Organization:
                    // Организация не может быть дочерней
                    context.ValidationResult.AddError(this, "$Mesages_OrganizationCantBeChild");
                    return default;
                case (int)DepartmentType.Branch: // филиал
                    // первое представительство выше по дереву
                    mainParentDepartmentTypeID = 2;
                    break;
                case (int)DepartmentType.Agency: // Представительство
                    // первая организация выше по дереву
                    mainParentDepartmentTypeID = 0;
                    break;
                default:
                    context.ValidationResult.AddError(this, "$Mesages_WrongDepartmentTypeID");
                    return default;
            }

            db
                .SetCommand(
                    builder
                        .With("DepartmentsCTE", dc =>
                                dc.Select()
                                        .C("rc", "ID", "ParentID").V(0).As("HierarchyLevel")
                                    .From("Roles", "rc").NoLock()
                                    .Where()
                                        .C("rc", "TypeID").Equals().V(2)
                                        .And()
                                        .C("rc", "ID").Equals().P("DepartmentdID")
                                    .UnionAll()
                                    .Select()
                                        .C("rp", "ID", "ParentID").C("dc", "HierarchyLevel").Add().V(1).As("HierarchyLevel")
                                    .From("Roles", "rp").NoLock()
                                    .InnerJoin("DepartmentsCTE", "dc")
                                        .On().C("dc", "ParentID").Equals().C("rp", "ID")
                                    .Where()
                                        .C("rp", "TypeID").Equals().V(2),
                            recursive: true)
                        .Select().Top(1)
                            .C("r", "ID", "Name")
                        .From("DepartmentsCTE", "dc")
                        .InnerJoin("Roles", "r").NoLock()
                            .On().C("r", "ID").Equals().C("dc", "ID")
                        .InnerJoin("AclDepartmentRoles", "acd").NoLock()
                            .On().C("acd", "ID").Equals().C("r", "ID")
                        .Where()
                            .If(mainParentDepartmentTypeID.HasValue, q => q.C("acd", "DepartmentTypeID").Equals().P("DepartmentTypeID"))
                            .If(!mainParentDepartmentTypeID.HasValue, q => q.C("acd", "DepartmentTypeID").IsNotNull())
                        .OrderBy("dc", "HierarchyLevel", SortOrder.Ascending)
                        .Limit(1).Build(),
                    db.Parameter("DepartmentdID", departmentID),
                    db.Parameter("DepartmentTypeID", mainParentDepartmentTypeID))
                .LogCommand();

            await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
            {
                if (await reader.ReadAsync(context.CancellationToken))
                {
                    var mainDepartmentID = reader.GetNullableGuid(0);
                    var mainDepartmentName = reader.GetNullableString(1);

                    if (mainDepartmentID == null)
                    {
                        return default;
                    }

                    return (mainDepartmentID.Value, mainDepartmentName);
                }
            }

            return default;
        }

        #endregion

        #region Base Overrides

        public override async Task BeforeCommitTransaction(ICardStoreExtensionContext context)
        {
            Card card;
            if ((card = context.Request.TryGetCard()) == null)
            {
                return;
            }

            if ((!card.Sections.TryGetValue("AclDepartmentRoles", out var aclDepartmentRolesSection) ||
                 !aclDepartmentRolesSection.RawFields.ContainsKey("DepartmentTypeID")) &&
                (!card.Sections.TryGetValue("Roles", out CardSection rolesSection) ||
                 !rolesSection.RawFields.ContainsKey("ParentID")))
            {
                return;
            }

            await using (context.DbScope.Create())
            {
                var db = context.DbScope.Db;
                var factory = context.DbScope.BuilderFactory;

                db
                    .SetCommand(
                        factory
                            .Select().Top(1)
                                .C("adr", "DepartmentTypeID")
                                .C("r", "ParentID")
                            .From("Roles", "r").NoLock()
                            .InnerJoin("AclDepartmentRoles", "adr").NoLock()
                                .On().C("r", "ID").Equals().C("adr", "ID")
                            .Where()
                                .C("r", "ID").Equals().P("DepartmentdID")
                            .Limit(1)
                            .Build(),
                        db.Parameter("DepartmentdID", card.ID))
                    .LogCommand();

                int? departmentTypeID = -1;
                await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
                {
                    if (await reader.ReadAsync(context.CancellationToken))
                    {
                        departmentTypeID = reader.GetNullableInt32(0);
                    }
                }

                if (!await UpdateCurrentDepartmentAsync(context, db, card, departmentTypeID))
                {
                    return;
                }

                // получение списка подразделений наже по дереву и их обновление
                db
                    .SetCommand(
                        context.DbScope.BuilderFactory.Create()
                            .With("RolesCTE", dc =>
                                dc.Select()
                                    .C("rp", "ID").V(0).As("HierarchyLevel")
                                .From("Roles", "rp").NoLock()
                                .Where()
                                    .C("rp", "TypeID").Equals().V(2)
                                .And()
                                    .C("rp", "ID").Equals().P("DepartmentdID")
                                .UnionAll()
                                .Select()
                                    .C("rc", "ID").C("rct", "HierarchyLevel").Add().V(1).As("HierarchyLevel")
                                .From("Roles", "rc").NoLock()
                                .InnerJoin("RolesCTE", "rct")
                                    .On().C("rc", "ParentID").Equals().C("rct", "ID")
                                .Where()
                                    .C("rc", "TypeID").Equals().V(2),
                                recursive: true)
                            .Select()
                                .C("rc", "ID")
                                .C("acr", "DepartmentTypeID")
                                .C("acr", "MainDepartmentID")
                            .From("RolesCTE", "rc")
                            .InnerJoin("AclDepartmentRoles", "acr").NoLock()
                                .On().C("acr", "ID").Equals().C("rc", "ID")
                            .Where()
                                .C("rc", "ID").NotEquals().P("DepartmentdID")
                            .OrderBy("rc", "HierarchyLevel", SortOrder.Ascending)
                            .Build(),
                        db.Parameter("DepartmentdID", card.ID))
                    .LogCommand();

                var departmentsToUpdate = new List<(Guid DepartmentID, int? DepartmentType, Guid? MainDepartmentID)>();
                await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
                {
                    while (await reader.ReadAsync(context.CancellationToken))
                    {
                        departmentsToUpdate.Add(
                            (reader.GetGuid(0), reader.GetNullableInt32(1), reader.GetNullableGuid(2)));
                    }
                }

                foreach (var department in departmentsToUpdate)
                {
                    // поиск нужного основного подразделения выше по дереву с проверкой - совпадает ли оно с текущим значением в проверяемом подразделении
                    var (mainDepartmentID, mainDepartmentName) = 
                        await this.GetMainDepartmentInfoAsync(
                            context, db, department.DepartmentID, department.DepartmentType);

                    if (mainDepartmentID is null
                        || mainDepartmentID == department.MainDepartmentID)
                    {
                        // Совпадает или отсутствует, обновлять не будем
                        continue;
                    }

                    // если не совпадает, то загрузка карточки подразделения и обновление
                    var getResponse = await this.cardRepository.GetAsync(
                        new CardGetRequest
                        {
                            CardID = department.DepartmentID,
                            RestrictionFlags = 
                                CardGetRestrictionFlags.RestrictFileSections |
                                CardGetRestrictionFlags.RestrictTaskCalendar |
                                CardGetRestrictionFlags.RestrictTaskHistory | 
                                CardGetRestrictionFlags.RestrictTasks | 
                                CardGetRestrictionFlags.RestrictTaskSections |
                                CardGetRestrictionFlags.RestrictFiles
                        },
                        context.CancellationToken);

                    if (!getResponse.ValidationResult.IsSuccessful())
                    {
                        context.ValidationResult.Add(getResponse.ValidationResult);
                        return;
                    }

                    var departmentCard = getResponse.Card;

                    departmentCard.Sections["AclDepartmentRoles"].Fields["MainDepartmentID"] = mainDepartmentID;
                    departmentCard.Sections["AclDepartmentRoles"].Fields["MainDepartmentName"] = mainDepartmentName;

                    departmentCard.RemoveAllButChanged();

                    var storeResponse = await this.cardRepository.StoreAsync(
                        new CardStoreRequest
                        {
                            Card = departmentCard
                        });

                    if (!storeResponse.ValidationResult.IsSuccessful())
                    {
                        context.ValidationResult.Add(storeResponse.ValidationResult);
                        return;
                    }
                }
            }
        }

        #endregion
    }
}
