﻿using System;
using System.Threading.Tasks;
using Tessa.Cards;
using Tessa.Cards.Extensions;
using Tessa.Platform.Data;
using Tessa.Platform.Validation;

namespace Tessa.Extensions.AclExamples.Server.Cards
{
    public class MeetingProtocolCardStoreExtension : CardStoreExtension
    {
        #region Base Overrides

        public override async Task BeforeRequestWhenTypeResolved(ICardStoreExtensionContext context)
        {
            Card card;
            if (!context.ValidationResult.IsSuccessful() ||
                (card = context.Request.TryGetCard()) == null ||
                !card.Sections.TryGetValue("AclMeetingProtocols", out var meetingProtocolsSection) ||
                !card.Sections.TryGetValue("DocumentCommonInfo", out var commonInfoSection))
            {
                return;
            }

            if (commonInfoSection.RawFields.TryGetValue("AuthorID", out object authorIDObject))
            {
                var authorID = (Guid?)authorIDObject;

                if (!authorID.HasValue)
                {
                    context.ValidationResult.AddError("$Mesages_AuthorMustSpecified");
                }
                else
                {
                    meetingProtocolsSection.RawFields.TryGetValue("AuthorDepartmentID", out object authorDepartmentIDObject);
                    meetingProtocolsSection.RawFields.TryGetValue("AuthorDepartmentName", out object authorDepartmentNameObject);
                    meetingProtocolsSection.RawFields.TryGetValue("AuthorOrganizationID", out object authorOrganizationIDObject);
                    meetingProtocolsSection.RawFields.TryGetValue("AuthorOrganizationName", out object authorOrganizationNameObject);

                    var authorDepartmentID = (Guid?) authorDepartmentIDObject;
                    var authorDepartmentName = (string)authorDepartmentNameObject;
                    
                    var authorOrganizationID = (Guid?)authorOrganizationIDObject;
                    var authorOrganizationName = (string)authorOrganizationNameObject;

                    if (!authorDepartmentID.HasValue)
                    {
                        context.ValidationResult.AddError("$Mesages_AuthorDepartmentMustSpecified");
                    }
                    else if (!authorOrganizationID.HasValue)
                    {
                        context.ValidationResult.AddError("$Mesages_AuthorOrganizationMustSpecified");
                    }
                    else
                    {
                        await using (context.DbScope.Create())
                        {
                            var db = context.DbScope.Db;
                            var factory = context.DbScope.BuilderFactory;

                            db
                                .SetCommand(
                                    factory
                                        .Select().Top(1)
                                            .C("r", "ID").As("AuthorDepartmentID")
                                            .C("r", "Name").As("AuthorDepartmentName")
                                            .C("org", "ID").As("AuthorOrganizationID")
                                            .C("org", "Name").As("AuthorOrganizationName")
                                        .From("RoleUsers", "ru").NoLock()
                                        .InnerJoin("Roles", "r").NoLock()
                                            .On().C("r", "ID").Equals().C("ru", "ID")
                                        .LeftJoinLateral(org=>org
                                            .Select().Top(1)
                                                .C("do", "ID", "Name")
                                            .From().Function("AeGetOrganization", b => b.C("r", "ID")).As("do")
                                            .Limit(1), "org")
                                        .Where()
                                            .C("r", "TypeID").Equals().V(2)
                                            .And()
                                            .C("r", "ID").Equals().P("AuthorID")
                                        .OrderBy("r", "ID")
                                        .Limit(1)
                                        .Build(),
                                    db.Parameter("AuthorID", authorID))
                                .LogCommand();

                            await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
                            {
                                if (await reader.ReadAsync(context.CancellationToken))
                                {
                                    var authorDepartmentIDFromServer = reader.GetNullableGuid(0);
                                    var authorDepartmentNameFromServer = reader.GetNullableString(1);

                                    var authorOrganizationIDFromServer = reader.GetNullableGuid(2);
                                    var authorOrganizationNameFromServer = reader.GetNullableString(3);

                                    if (authorDepartmentID != authorDepartmentIDFromServer ||
                                        authorDepartmentName != authorDepartmentNameFromServer)
                                    {
                                        context.ValidationResult.AddError("$Mesages_WrongAuthorDepartment");
                                    }
                                    if (authorOrganizationID != authorOrganizationIDFromServer ||
                                        authorOrganizationName != authorOrganizationNameFromServer)
                                    {
                                        context.ValidationResult.AddError("$Mesages_WrongAuthorOrganization");
                                    }
                                }
                            }
                        }
                    }
                }
            }

            if (meetingProtocolsSection.RawFields.TryGetValue("CollegiatePartID", out object collegiatePartIDObject))
            {
                var collegiatePartID = (Guid?) collegiatePartIDObject;

                if (!collegiatePartID.HasValue)
                {
                    context.ValidationResult.AddError("$Mesages_CollegiatePartMustSpecified");
                }
                else
                {
                    meetingProtocolsSection.RawFields.TryGetValue("CollegiatePartOrganizationID", out object collegiatePartOrganizationIDObject);
                    meetingProtocolsSection.RawFields.TryGetValue("CollegiatePartOrganizationName", out object collegiatePartOrganizationNameObject);

                    var collegiatePartOrganizationID = (Guid?)collegiatePartOrganizationIDObject;
                    var collegiatePartOrganizationName = (string)collegiatePartOrganizationNameObject;


                    if (!collegiatePartOrganizationID.HasValue)
                    {
                        context.ValidationResult.AddError("$Mesages_СollegiatePartOrganizationMustSpecified");
                    }
                    else
                    {
                        await using (context.DbScope.Create())
                        {
                            var db = context.DbScope.Db;
                            var factory = context.DbScope.BuilderFactory;

                            db
                                .SetCommand(
                                    factory
                                        .Select().Top(1)
                                            .C("cp", "OrganizationID").As("CollegiatePartOrganizationID")
                                            .C("cp", "OrganizationName").As("CollegiatePartOrganizationName")
                                        .From("AclCollegiatePart", "cp").NoLock()
                                        .Where()
                                            .C("cp", "ID").Equals().P("СollegiatePartID")
                                        .Limit(1)
                                        .Build(),
                                    db.Parameter("СollegiatePartID", collegiatePartID))
                                .LogCommand();

                            await using (var reader = await db.ExecuteReaderAsync(context.CancellationToken))
                            {
                                if (await reader.ReadAsync(context.CancellationToken))
                                {
                                    var сollegiatePartOrganizationIDFromServer = reader.GetNullableGuid(0);
                                    var сollegiatePartOrganizationNameFromServer = reader.GetNullableString(1);

                                    if (collegiatePartOrganizationID != сollegiatePartOrganizationIDFromServer ||
                                        collegiatePartOrganizationName != сollegiatePartOrganizationNameFromServer)
                                    {
                                        context.ValidationResult.AddError("$Mesages_WrongCollegiatePartOrganization");
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        #endregion
    }
}
